/*
 * Percepio DFM v2.1.0
 * Copyright 2023 Percepio AB
 * www.percepio.com
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * DFM serial port Cloud port
 */

#include <stddef.h>
#include <dfmCloudPort.h>
#include <dfmCloudPortConfig.h>
#include <dfm.h>
#include <string.h>
#include <stdio.h>

#if (defined(DFM_CFG_ENABLED) && ((DFM_CFG_ENABLED) >= 1))

/* Prototype for the print function */
extern void vMainUARTPrintString( char * pcString );

#define DFM_PRINT_SERIAL_DATA(msg) printf(msg)

static DfmCloudPortData_t *pxCloudPortData = (void*)0;

static uint32_t prvPrintDataAsHex(uint8_t* data, int size);
static DfmResult_t prvSerialPortUploadEntry(DfmEntryHandle_t xEntryHandle);

static uint32_t prvPrintDataAsHex(uint8_t* data, int size)
{
	uint32_t checksum = 0;
	int i;
	char buf[10];

    for (i = 0; i < size; i++)
    {
    	uint8_t byte = data[i];
    	checksum += byte;
        snprintf(buf, sizeof(buf), " %02X", (unsigned int)byte);

        if (i % 20 == 0)
        {
            DFM_CFG_LOCK_SERIAL();
            DFM_PRINT_ALERT_DATA(("[[ DATA:"));
        }

        DFM_PRINT_ALERT_DATA(buf);

        if ( (i+1) % 20 == 0)
        {
            DFM_PRINT_ALERT_DATA((" ]]" LNBR));
            DFM_CFG_UNLOCK_SERIAL();            
        }
    }

    if (i % 20 != 0)
    {
        DFM_PRINT_ALERT_DATA((" ]]" LNBR));
        DFM_CFG_UNLOCK_SERIAL();
    }

    return checksum;
}

static DfmResult_t prvSerialPortUploadEntry(DfmEntryHandle_t xEntryHandle)
{
	uint32_t datalen;

	if (pxCloudPortData == (void*)0)
	{
		return DFM_FAIL;
	}

	if (xEntryHandle == 0)
	{
		return DFM_FAIL;
	}

	if (xDfmEntryGetSize(xEntryHandle, &datalen) == DFM_FAIL)
	{
		return DFM_FAIL;
	}

	if (datalen > 0xFFFF)
	{
		return DFM_FAIL;
	}

	DFM_CFG_LOCK_SERIAL();
	DFM_PRINT_SERIAL_DATA(LNBR "[[ DevAlert Data Begins ]]" LNBR);
	DFM_CFG_UNLOCK_SERIAL();

	(void) prvPrintDataAsHex((uint8_t*)xEntryHandle, datalen);

    // Checksum not provided (0) since not updated for the new Receiver script (uses a different checksum algorithm). If 0, checksum is ignore.
	snprintf(pxCloudPortData->buf, sizeof(pxCloudPortData->buf), "[[ DevAlert Data Ended. Checksum: %d ]]" LNBR, (unsigned int)0);

	DFM_CFG_LOCK_SERIAL();
	DFM_PRINT_SERIAL_DATA(pxCloudPortData->buf);
	DFM_CFG_UNLOCK_SERIAL();
                
	return DFM_SUCCESS;
}

DfmResult_t xDfmCloudPortInitialize(DfmCloudPortData_t* pxBuffer)
{
	pxCloudPortData = pxBuffer;

	return DFM_SUCCESS;
}

DfmResult_t xDfmCloudPortSendAlert(DfmEntryHandle_t xEntryHandle)
{
	return prvSerialPortUploadEntry(xEntryHandle);
}

DfmResult_t xDfmCloudPortSendPayloadChunk(DfmEntryHandle_t xEntryHandle)
{
	return prvSerialPortUploadEntry(xEntryHandle);
}

#endif
