from http.server import SimpleHTTPRequestHandler, ThreadingHTTPServer
import os, platform, shutil
from pathlib import Path
import sys
import userhooks
from struct import unpack
import re
from http import HTTPStatus
import subprocess
from typing import Optional
from urllib.parse import urlparse, unquote, parse_qs



def read_alert_header_fields(filename: str):
    """
    Read Product ID, Revision, Device Name, and Description from an ALERT header file.

    Input file must be the inner ALERT header payload written by bin2alerts.py
    (begins with b'PDfm'). The device name is NOT in this payload; we derive it
    from the path ".../DevAlert/<device>/<session>/<alert>/<file>" if present.

    Returns: (product_id: int, revision: str, device_name: str|None, description: str)
    """
    with open(filename, "rb") as f:
        data = f.read()

    if len(data) < 20:
        raise ValueError("File too short for ALERT header")
    if data[0:4] != b"PDfm":
        raise ValueError("Not an ALERT header (missing 'PDfm' magic)")

    # Endianness marker at [4:6], same mapping style as your read_payload_size()
    marker = data[4:6]
    if marker == b"\x0F\xF0":
        endian = ">"   # big-endian
    elif marker == b"\xF0\x0F":
        endian = "<"   # little-endian
    else:
        raise ValueError(f"Unknown endianness marker: {marker.hex()}")

    # Single-byte fields (endianness irrelevant for these)
    fw_sz   = data[7]   # ucFirmwareVersionSize
    max_sym = data[8]   # ucMaxSymptoms
    desc_sz = data[10]  # ucDescriptionSize

    # Product ID (ulProduct) at [12..15], endianness applies
    product_id = unpack(endian + "I", data[12:16])[0]

    # Skip ulAlertType (next 4 bytes) and the full symptoms array (max_sym * 8 bytes)
    offset = 20 + (max_sym * 8)

    # Firmware version string
    if len(data) < offset + fw_sz:
        raise ValueError("File too short for firmware version string")
    fw_bytes = data[offset:offset + fw_sz]
    revision = fw_bytes.split(b"\x00", 1)[0].decode("ascii", errors="ignore")
    offset += fw_sz

    # Description string
    if len(data) < offset + desc_sz:
        raise ValueError("File too short for description string")
    desc_bytes = data[offset:offset + desc_sz]
    description = desc_bytes.split(b"\x00", 1)[0].decode("ascii", errors="ignore")

    # Device name is not in the inner ALERT header; derive from path if possible
    device_name = None
    norm = filename.replace("\\", "/")
    parts = [p for p in norm.split("/") if p]
    for i, seg in enumerate(parts):
        if seg == "DevAlert" and i + 1 < len(parts):
            device_name = parts[i + 1]
            break

    return str(product_id), revision, device_name, description


def read_payload_size(filename: str) -> int:
        """
        Read usChunkCount from a binary header:
        [0..3]=cStartMarker[4], [4..5]=usEndianess, one 16-bit field to skip,
        then [8..11]=payload_size.
        Endianness marker:
          0x0F 0xF0 -> little endian (Arm Cortex-M example)
          0xF0 0x0F -> big endian (other)
        """
        with open(filename, "rb") as f:
            hdr = f.read(16)
        if len(hdr) < 16:
            raise ValueError("File too short for header")

        marker = hdr[4:6]
        if marker == b"\x0F\xF0":
            endian = ">"  # big endian
        elif marker == b"\xF0\x0F":
            endian = "<"  # little endian
        else:
            raise ValueError(f"Unknown endianness marker: {marker.hex()}")

        return unpack(endian + "I", hdr[8:12])[0]

class BadRequestException(Exception): pass
class DownloadException(Exception): pass


# reuse your existing SAFE_PATH_RE (allows leading slash)
SAFE_PATH_RE = re.compile(r'^[A-Za-z0-9._/\-]+$')
MAX_PATH_LEN = 4096
MAX_SEGMENT_LEN = 512

def parse_and_validate(raw_path: str, handler):
    """
    Parse raw_path (may include ?query), percent-decode, validate path & query.
    Returns (decoded_path:str, query:dict) on success, or (None, None) after sending 400 on failure.
    """
    # split path and query safely
    parsed = urlparse(raw_path)   # parsed.path, parsed.query
    # percent-decode path
    try:
        decoded = unquote(parsed.path)
    except Exception:
        handler.send_error(HTTPStatus.BAD_REQUEST, "Bad request: invalid percent-encoding")
        return None, None

    # basic length checks
    if len(decoded) > MAX_PATH_LEN:
        handler.send_error(HTTPStatus.BAD_REQUEST, "Bad request: path too long")
        return None, None

    # disallow path traversal
    if ".." in decoded:
        handler.send_error(HTTPStatus.BAD_REQUEST, "Bad request: path traversal attempt")
        return None, None

    # whitelist characters for decoded path
    if not SAFE_PATH_RE.match(decoded):
        handler.send_error(HTTPStatus.BAD_REQUEST, "Bad request: illegal characters in URL path")
        return None, None

    # per-segment length
    for seg in decoded.split("/"):
        if len(seg) > MAX_SEGMENT_LEN:
            handler.send_error(HTTPStatus.BAD_REQUEST, "Bad request: path segment too long")
            return None, None

    # validate query string: allow empty or exactly ?forcereload=<digits>
    qs = parse_qs(parsed.query, keep_blank_values=True)
    if qs:
        if len(qs) != 1 or "forcereload" not in qs:
            handler.send_error(HTTPStatus.BAD_REQUEST, "Bad request: unexpected query parameters")
            return None, None
        vals = qs["forcereload"]
        if len(vals) != 1 or not vals[0].isdigit():
            handler.send_error(HTTPStatus.BAD_REQUEST, "Bad request: illegal forcereload value")
            return None, None

    return decoded, qs


def run_safer(path: str, *args: str):
    """
    Start a child process in the background (non-detached), inheriting stdout/stderr.

    - Safer alternative to os.system(): avoids shell=True
    - Inherits stdout/stderr so output appears in same terminal.
    - Caller is responsible for polling/waiting to avoid zombies.
    - Automatically resolves executables on PATH if 'path' is not absolute.

    Example:
        run_safer("python", "-m", "http.server", "8000")
    """
    # Resolve executable on PATH if not absolute
    if not os.path.isabs(path):
        found = shutil.which(path)
        if found:
            path = found

    cmd = [path, *map(str, args)]
    print("Executing:", " ".join(cmd))

    try:
        proc = subprocess.Popen(
            cmd,
            stdout=None,               # inherit from parent
            stderr=None,               # inherit from parent
            close_fds=True,            # avoid leaking file descriptors
            start_new_session=False    # remain in same process group/session
        )
        return proc  # return process handle for optional wait/poll
    except FileNotFoundError:
        print(f"Executable not found: {path}")
    except PermissionError:
        print(f"Permission denied executing: {path}")
    except Exception as e:
        print(f"Failed to start command: {cmd} ({e})")
        


class HTTPHandler(SimpleHTTPRequestHandler):

    # Input validation
    def do_GET(self):
        decoded_path, query = parse_and_validate(self.path, self)
        if decoded_path is None:
            return

        self.path = decoded_path
        return super().do_GET()

    # Input validation
    def do_HEAD(self):
        decoded_path, query = parse_and_validate(self.path, self)
        if decoded_path is None:
            return
        self.path = decoded_path
        return super().do_HEAD()

    def translate_path(self, path):
        # This is called on each (input validated) HTTP request. 
        # Note that the returned file is normally not displayed in the Detect dashboard since background HTTP requests.

        print("Request: " + path)
        
        if path.startswith("/config"):
            run_safer(dispatcher_path, "--settings-file", "dispatcher-settings.json")
            
            return base_path + "index.html"

        if path.startswith("/tz-user-manual"):
            helpPath = path.replace("/tz-user-manual", "/home/app/Tracealyzer/help")
            print ("Modified path to: " + helpPath)
            return helpPath

        if path.startswith("/tz"):
            run_safer(tz_path)
            return base_path + "index.html"

        # Link would be e.g. http://localhost:8085/devalert/issue/DevAlert/D-004A003C-464B5010-20333253/S-8DBC0933-BCE717DA-10004A50/1/da_payload2
        # Argument in https server then becomes /devalert/issue/DevAlert/D-004A003C-464B5010-20333253/S-8DBC0933-BCE717DA-10004A50/1/da_payload2
        # Dispatcher expects percepio://devalert/issue/DevAlert/D-004A003C-464B5010-20333253/S-8DBC0933-BCE717DA-10004A50/1/da_payload2
        # So we only need to add "percepio:/".
                     
        # Handle DevAlert requests
        if path.startswith("/devalert"):
            
            try:

                request = path.removeprefix("/devalert/issue")

                # Split the request path into "directory" (alert key) and "filename"
                alert_key, requested_payload = os.path.split(request)
                
                # The base path where the alert files should be stored
                local_alert_dir = os.environ['ALERTS_DIR'].strip('"')
                
                # First we need to know the payload number (1, 2, possibly a few more)
                # This is the last character in the requested payload name.
                if requested_payload[-1].isdigit():
                    payload_number_str = requested_payload[-1]
                else:
                    raise BadRequestException("Requested payload didn't end with a number (" + requested_payload + ")")
                
                                        
                # Always call read_alert_header_fields(), even if the file is local. 
                alert_header_filename = "1-1_da_header"
                if (userhooks.enable_download_alert_hook):
                    if (userhooks.download_alert_file(
                            alert_header_filename, 
                            alert_key, 
                            local_alert_dir + alert_key) != True):
                        raise DownloadException("download_alert_file hook didn't return \"success\" (= True)") 
                                        
                alert_header_path = local_alert_dir + alert_key + "/" + alert_header_filename                
 
                # Extract metadata fields from the alert header file
                product_id, revision, device_name, description = read_alert_header_fields(alert_header_path)
                    
                # If to download the alert payload (not locally available)
                if (userhooks.enable_download_alert_hook):
                
                    # Download the payload header and read the total payload size
                    payload_header_filename = "1-1_da_payload" + payload_number_str + "_header"
                
                    if (userhooks.download_alert_file(
                            payload_header_filename, 
                            alert_key, 
                            local_alert_dir + alert_key) != True):
                        raise DownloadException("download_alert_file hook didn't return \"success\" (= True)") 
                
                    payload_header_path = local_alert_dir + alert_key + "/" + payload_header_filename                
                    total_payload_size = read_payload_size(payload_header_path)
                
                    # Download the first chunk and check its size.
                    # If not equal to total payload size, this is the max chunk size
                    # From that, we can calculate the expected chunk count and thus 
                    # provide a list of filenames to download.
                
                    # Define parameters and download the first file
                    first_chunk_filename = "1_da_payload" + payload_number_str 

                    if (userhooks.download_alert_file(
                            first_chunk_filename, 
                            alert_key, 
                            local_alert_dir + alert_key) != True):
                        raise DownloadException("download_alert_file hook didn't return \"success\" (= True)") 
                
                    first_chunk_path = local_alert_dir + alert_key + "/" + first_chunk_filename
                    first_chunk_size = os.path.getsize(first_chunk_path)
                
                    if (first_chunk_size != total_payload_size):
                        chunk_count = total_payload_size // first_chunk_size + 1
                        print("  Downloading " + str(chunk_count-1) + " additional file(s)")
                        
                        i = 2 # The first is already downloaded
                        while (i <= chunk_count):
                            chunk_filename = str(i) + "_da_payload" + payload_number_str
                            chunk_path = local_alert_dir + alert_key + "/" + first_chunk_filename
                            if (userhooks.download_alert_file(
                                    chunk_filename, 
                                    alert_key, 
                                    local_alert_dir + alert_key) != True):
                                raise DownloadException("download_alert_file hook didn't return \"success\" (= True)") 
                            i+=1
                        
                if (userhooks.enable_download_elf_hook):
                
                    if (platform.system() == 'Windows'):
                        local_elf_path = "cache/download/prod_" + product_id + "/" + revision + "/image.elf"
                        os.environ['ELF_PATH'] = local_elf_path
                    elif (platform.system() == 'Linux'):
                        local_elf_path = "../../home/app/.config/PercepioDevAlertDispatcher/Cache/prod_" + product_id + "/" + revision + ".elf"
                        os.environ['ELF_REL_PATH'] = local_elf_path
                    
                    local_elf_dir, sddsd = os.path.split(local_elf_path)
                    print("Creating ELF folder " + local_elf_dir)
                    Path(local_elf_dir).mkdir(parents=True, exist_ok=True)
                    if (userhooks.download_elf_file(product_id, revision, local_elf_path) != True):
                        raise DownloadException("download_elf_file hook didn't return \"success\" (= True)")
                
                if (userhooks.enable_download_src_hook):
                    if (platform.system() == 'Windows'):
                        local_src_dir = "cache/download/prod_" + product_id + "/" + revision + "/src"
                        os.environ['SRC_PATH'] = local_src_dir
                    elif (platform.system() == 'Linux'):
                        local_src_dir = "../../home/app/.config/PercepioDevAlertDispatcher/Cache/prod_" + product_id + "/" + revision + "/src/"
                        os.environ['SRC_REL_PATH'] = local_src_dir
                    
                    print("Creating SRC folder " + local_src_dir)
                    Path(local_src_dir).mkdir(parents=True, exist_ok=True)
                    if (userhooks.download_src_files(product_id, revision, local_src_dir) != True):
                        raise DownloadException("download_src_files hook didn't return \"success\" (= True)")

            except BadRequestException as e:
                print(f"  Error: {e}\n")
                return base_path + "index.html"

            except DownloadException as e:
                print(f"  Error: {e}\n")
                return base_path + "index.html"

            except FileNotFoundError as e:
                print(f"  Error: {e}\n")
                return base_path + "index.html"
                
            # Invoke Dispatcher on the request. Needs "percepio://" before the alert key (= path)
            # ensure no leading whitespace or extra slashes
            payload_arg = "percepio://" + path.lstrip('/')
            proc = run_safer(dispatcher_path, "--settings-file", "dispatcher-settings.json", payload_arg)


            return base_path + "index.html"
        
        return base_path + path


    def end_headers(self):
        self.send_my_headers()
        SimpleHTTPRequestHandler.end_headers(self)

    def send_my_headers(self):
        self.send_header("Access-Control-Allow-Origin", "*")
        self.send_header('Access-Control-Allow-Methods', 'GET')
        self.send_header('Access-Control-Allow-Headers', 'Content-Type')
        self.send_header('Cache-Control', 'No-Cache, No-Store, Max-Age=0')
        self.send_header('Connection', 'close')

# Use ThreadingHTTPServer to allow multithreaded request handling
class HTTPServer(ThreadingHTTPServer):

    def __init__(self, base_path, server_address, RequestHandlerClass=HTTPHandler):
        self.base_path = base_path
        super().__init__(server_address, RequestHandlerClass)

# Create and run the server

base_path = ""
tz_path = ""

host_os = platform.system()

if (host_os == 'Windows'):

    base_path = "content/"
    tz_path = "tracealyzer/Tracealyzer.exe"    
    dispatcher_path = "dispatcher/DevAlertDispatcher.exe"
    
    # No longer copying settings.json, provided as argument (--settings-file) in dispatcher calls.
                
elif (host_os == 'Linux'): 
    base_path = "/home/app/http/content/"
    tz_path = "/home/app/tz/launch-tz.sh"
    dispatcher_path = "/home/app/da-dispatcher/DevAlertDispatcher"
    
    # This assumes Docker - if not available inside the container, define it so the code above works.
    os.environ['ALERTS_DIR'] = "/host-mounts/alert_data"
    
    # No need to copy settings.json, since included in the Docker image at the expected location

else:
    print("This version of the Percepio Detect Client is only for Windows or Linux.")
    quit()

# Allows for customized initialization of the client.
# This can be useful on the docker-based Linux client, e.g. for reading a password over stdin.
if (userhooks.enable_start_hook):
    userhooks.once_on_start()

# Only accepts localhost connections
httpd = HTTPServer(base_path, ('127.0.0.1', 8085))

print("Ready.")
httpd.serve_forever()
