#!/usr/bin/python3

# bin2alerts - extracts alert files from a raw binary file
# containing DFM data. The input file must start with the
# first DFM entry. Certain checks are also made to verify
# the DFM data structure. Use the verbose option to
# pretty-print the DFM data.

import argparse
import os
import io
import sys
import time
import math
import binascii
from binascii import unhexlify
from pathlib import Path

def crc16_ccit(data: bytearray):
    seed = 0x0000
    for i in range(0, len(data)):
        cur_byte = data[i]
        e = (seed ^ cur_byte) & 0xFF
        f = (e ^ (e << 4)) & 0xFF
        seed = (seed >> 8) ^ (f << 8) ^ (f << 3) ^ (f >> 4)

    return seed

# Defines from DFM
DFM_ENTRY_TYPE_ALERT = 0x1512
DFM_ENTRY_TYPE_PAYLOAD_HEADER = 0x4618
DFM_ENTRY_TYPE_PAYLOAD_CHUNK = 0x8371

typeNames = {}
typeNames[DFM_ENTRY_TYPE_ALERT] = 'DFM_ENTRY_TYPE_ALERT'
typeNames[DFM_ENTRY_TYPE_PAYLOAD_HEADER] = 'DFM_ENTRY_TYPE_PAYLOAD_HEADER'
typeNames[DFM_ENTRY_TYPE_PAYLOAD_CHUNK] = 'DFM_ENTRY_TYPE_PAYLOAD_CHUNK'

endian = 'little'

_input_file = None

print_indent = 0

verbose_mode = False

exit_at_eof = True

session_id = None

alert_data = bytearray()

def generate_topic(entry_type, device_name, session_id, alert_id, chunk_index, chunk_count, entry_id) -> str:
    topic_string: str

    if entry_type == DFM_ENTRY_TYPE_ALERT:
        topic_string = "DevAlert/{}/{}/{}/{}-{}_da_header".format(
            device_name,
            session_id,
            alert_id,
            chunk_index,
            chunk_count
        )
    elif entry_type == DFM_ENTRY_TYPE_PAYLOAD_HEADER:
        topic_string = "DevAlert/{}/{}/{}/{}-{}_da_payload{}_header".format(
            device_name,
            session_id,
            alert_id,
            chunk_index,
            chunk_count,
            entry_id
        )
    elif entry_type == DFM_ENTRY_TYPE_PAYLOAD_CHUNK:
        topic_string = "DevAlert/{}/{}/{}/{}_da_payload{}".format(
            device_name,
            session_id,
            alert_id,
            chunk_index,            
            entry_id
        )
    else:
        print("Error, entry type not recognozed!")
        sys.exit(4)

    return topic_string

def pretty_print_bytes(byte_data, bytes_per_line=16):
    # Iterate through the byte data in chunks of 'bytes_per_line'
    for i in range(0, len(byte_data), bytes_per_line):
        # Slice out the current chunk of bytes
        chunk = byte_data[i:i + bytes_per_line]
        
        # Convert the bytes to a space-separated string of hex values
        hex_values = ' '.join(f'{byte:02X}' for byte in chunk)
        
        # Convert the bytes to a string, showing printable ASCII characters or '.' for non-printable characters
        ascii_values = ''.join(chr(byte) if 32 <= byte <= 126 else '.' for byte in chunk)
        
        # Print the formatted line
        print(f"    {i:04X}  {hex_values:<47}  {ascii_values}")

def set_input_file(file_path):

    global _input_file

    if file_path is None:
        _input_file = io.BufferedReader(sys.stdin.buffer)
    else:
        _input_file = open(file_path, "rb")


def readbytes(n):
    buf = bytearray()
    while len(buf) < n:
        chunk = _input_file.read(n - len(buf))
        if chunk:
            buf.extend(chunk)
        else:
            if exit_at_eof:
                sys.exit(0)
            else:
                time.sleep(0.1)
    
    alert_data.extend(buf)
    return bytes(buf)                

def read_uint32():
    return int.from_bytes(readbytes(4), endian)

def read_uint16():
    return int.from_bytes(readbytes(2), endian)

def read_uint8():
    return int.from_bytes(readbytes(1), endian)
    
def hex32(val):
    return f"0x{val:08X}"

def hex16(val):
    return f"0x{val:04X}"

def hex8(val):
    return f"0x{val:02X}"

def check_and_print(label, datatype, expected=None, length=0):
    global print_indent
    if (label):
        pad = ' ' * print_indent
        label = pad + label + ":"
        label = label.ljust(28, ' ')

    if (datatype == "u32"):
         val = read_uint32()
         s = label + hex32(val) + " (" + str(val) + ")"
    elif (datatype == "u16"):
         val = read_uint16()
         s = label + hex16(val) + " (" + str(val) + ")"
    elif (datatype == "u8"):
         val = read_uint8()
         s = label + hex8(val) + " (" + str(val) + ")"
    elif (datatype == "string"):
         # Just a sanity check. Not sure what the largest allowed payload is.
         if(length > 0 and length < (256*1024)):
             b = readbytes(length)
             val = b.decode('ascii', errors='ignore').rstrip('\x00')
             s = label + val
         else:
             print("Invalid length provided for readbytes (" + length + ")")
             sys.exit(3)
    elif (datatype == "data"):
         if (label): print(label)
         b = readbytes(length)

         if (verbose_mode):
             pretty_print_bytes(b)

         s = None
         val = b
    else:
         print("Datatype " + datatype + " not implemented.")
         sys.exit(2)

    if (verbose_mode and s):
        print(s, end=' ')

    if (expected):
        if isinstance(expected, list):
            if (not val in expected): 
                print("\n    Error: Expected one of the following values: " + str(expected))
                sys.exit(1)
            else:
                global typeNames
                if (verbose_mode): print("= " + typeNames[val])
                s = None
        else:
            if (not val == expected): 
                print("\n    Error: Expected value was " + str(expected))
                sys.exit(1)

    if (verbose_mode and s):
        print("")
        
    return val

def peekbytes(n :int) -> bytes:
    buf = bytearray()
    while True:
        chunk = _input_file.peek(n)
        if len(chunk) >= n:
            return chunk[:n]

        # Not enough buffered. If absolutely nothing is available, we may be at EOF or just waiting for producer.
        if len(chunk) == 0 and exit_at_eof:
            # Likely EOF: exit as requested
            sys.exit(0)

        # Wait a bit and try again (producer may write more)
        time.sleep(0.1)          

def read_byte_and_discard():
    while True:
        chunk = _input_file.read(1)
        if chunk:
            return
        if exit_at_eof:
            sys.exit(0)
        time.sleep(0.1)

def find_start_marker():
    pattern=b"\xD1\xD2\xD3\xD4"
    patlen = len(pattern)
    while True:
        
        # Look ahead without consuming. This always returns the requested number of bytes. 
        # If not available, it waits for more data or exits, depending on --eof arg.
        ahead = peekbytes(patlen)
        if ahead == pattern:
            return
        read_byte_and_discard()


from argparse import RawTextHelpFormatter

if __name__ == "__main__":
    parser = argparse.ArgumentParser(
        prog='bin2alerts',
        description='Extracts DFM alert files from a binary file, with certain controls to verify the data structure. Also allows for pretty-printing all the data with -v flag.',
        formatter_class=RawTextHelpFormatter     
    )
    parser.add_argument('-i', '--inputfile',   type=str, help='The log file to read, containing DFM data. If not provided, stdin is used.')
    parser.add_argument('-f', '--folder',      type=str, help='The folder where the output data should be saved.', required=False, default='alert-files')
    parser.add_argument('-e', '--eof',         type=str, help="What to do at end of file:\n    wait: keeps waiting for more data (exit using Ctrl-C).\n    exit: exits directly at end of file (default).", required=False, default='exit')
    parser.add_argument('-d', '--device_name', type=str, help="Sets the Device name, replacing the name set in the target-side DFM library.", required=False)
    parser.add_argument('-v', '--verbose',     action='store_true', help='Enables verbose output for verification and debugging.')

    # Tolerated but ignored args, allows for passing bin2alerts and txt2bin the same args from a shared top-level script.
    parser.add_argument('-o', '--outputfile', type=str, help='Not used. Added for argument compatibility between scripts.', required=False)

    args = parser.parse_args()
    
    if (args.verbose):
        print("bin2alerts.py")
        
        if (args.inputfile):
            print("  inputfile: " + args.inputfile)
        else:
            print("  inputfile: stdin")
        
        print("  folder: " + args.folder)
        
        print("  eof: " + args.eof)
        if (args.inputfile):
            print("  device_name: " + args.device_name)
        else:
            print("  device_name: -")

    accumulated_payload = bytes([])
    
    session_id = ""
    device_id = ""
    alert_counter = 0;

    verbose_mode = args.verbose

    if (verbose_mode): print("Verbose mode, showing all DFM data fields.")

    if (args.eof == 'exit'): exit_at_eof = True
    elif (args.eof == 'wait'): exit_at_eof = False
    else: sys.exit("Unrecognized option for --eof argument. Valid options are: wait, exit.")

    set_input_file(args.inputfile)

    counter = 0
    maxcount = 3333

    if (verbose_mode):
        print("---------------------------------------------------------------------------")


    while True:
        print_indent = 2

        # Reads and discards initial data from stdio, until at start marker. Respects --eof argument.
        find_start_marker()

        # Start marker is always in this order, regardless of endianess.
        check_and_print("cStartMarker[0]","u8", expected=0xD1)       
        check_and_print("cStartMarker[1]","u8", expected=0xD2)
        check_and_print("cStartMarker[2]","u8", expected=0xD3)
        check_and_print("cStartMarker[3]","u8", expected=0xD4)
        check_and_print("usEndianess","u16")
        check_and_print("usVersion","u16")

        entryType = check_and_print("usType", "u16", expected=[DFM_ENTRY_TYPE_ALERT, DFM_ENTRY_TYPE_PAYLOAD_HEADER, DFM_ENTRY_TYPE_PAYLOAD_CHUNK])
        entry_id = check_and_print("usEntryId", "u16")
        chunk_index = check_and_print("usChunkIndex", "u16")
        chunk_count = check_and_print("usChunkCount", "u16")
        alert_id = check_and_print("ulAlertId","u32")
        session_id_size = check_and_print("usSessionIdSize", "u16")
        device_name_size = check_and_print("usDeviceNameSize", "u16")
        description_size = check_and_print("usDescriptionSize", "u16")
        check_and_print("usReserved", "u16")
        data_size = check_and_print("ulDataSize", "u32")

        tmp = check_and_print("Session ID", "string", length=session_id_size)
        
        if (entryType == DFM_ENTRY_TYPE_ALERT):
            if (tmp == "$DUMMY_SESSION_ID"):
                # If session_id not provided in DFM data, use a host timestamp.
                # Use the same for all chunks in the same alert, until the next new alert.
                ts = math.trunc(time.time()*1000)
                session_id = str(ts)
                # To ensure alerts read in a batch don't get the same timestamp.
                time.sleep(0.01)
            else:
                session_id = tmp
  
        device_name = check_and_print("Device name", "string", length=device_name_size)
        if (device_name == "$DUMMY_DEVICE_ID"):
            if (args.device_name):
                device_name = args.device_name
            else:
                print("\n  Warning: Device name not specified in the arguments or in the DFM data .\n  Add argument -d <name> to set the device name, or set in the DFM library.\n")
  
        topic_string = generate_topic(entryType, device_name, session_id, alert_id, chunk_index, chunk_count, entry_id)

        if (verbose_mode):
            print("  Topic (relative path):    " + topic_string)
            print("  Output file:              " + os.path.join(args.folder, topic_string))
        else:
            print("  Creating file " + os.path.join(args.folder, topic_string))

        check_and_print("Description", "string", length=description_size)


        print_indent = 4

        # All check_and_print calls below is also acheck_and_printending the data to alert_data.
        # This is reset here, and written to the output file (topic_string) after handling these three entry types.
        alert_data = bytearray()

        if (entryType == DFM_ENTRY_TYPE_ALERT):
            if (verbose_mode): print("  ALERT_HEADER data:")

            check_and_print("ucStartMarkers[0]","u8", expected=0x50)       
            check_and_print("ucStartMarkers[1]","u8", expected=0x44)
            check_and_print("ucStartMarkers[2]","u8", expected=0x66)
            check_and_print("ucStartMarkers[3]","u8", expected=0x6D)
            check_and_print("usEndianess","u16")
            check_and_print("ucVersion","u8")
            ucFirmwareVersionSize = check_and_print("ucFirmwareVersionSize","u8")
            ucMaxSymptoms = check_and_print("ucMaxSymptoms","u8")
            check_and_print("ucSymptomCount","u8")
            ucDescriptionSize = check_and_print("ucDescriptionSize","u8")
            check_and_print("ucReserved0","u8")
            check_and_print("ulProduct","u32")
            check_and_print("ulAlertType","u32")
            for i in range(ucMaxSymptoms):
                check_and_print("Symptom ID " + str(i),"u32")
                check_and_print("Symptom Value " + str(i),"u32")

            check_and_print("cFirmwareVersionBuffer","string", length=ucFirmwareVersionSize)
            check_and_print("cAlertDescription","string", length=ucDescriptionSize)
            check_and_print("ucEndMarkers[0]","u8", expected=0x6D)       
            check_and_print("ucEndMarkers[1]","u8", expected=0x66)
            check_and_print("ucEndMarkers[2]","u8", expected=0x44)
            check_and_print("ucEndMarkers[3]","u8", expected=0x50)
            check_and_print("ulChecksum","u32")

        if (entryType == DFM_ENTRY_TYPE_PAYLOAD_HEADER):
            if (verbose_mode): print("  PAYLOAD_HEADER data:")

            check_and_print("ucStartMarkers[0]","u8", expected=0x50)       
            check_and_print("ucStartMarkers[1]","u8", expected=0x44)
            check_and_print("ucStartMarkers[2]","u8", expected=0x61)
            check_and_print("ucStartMarkers[3]","u8", expected=0x50)
            check_and_print("usEndianess","u16")
            check_and_print("ucVersion","u8")
            ucFilenameSize = check_and_print("ucFilenameSize","u8")
            ulFileSize = check_and_print("ulFileSize","u32")
            check_and_print("cFilenameBuffer","string", length=ucFilenameSize)
            check_and_print("ucEndMarkers[0]","u8", expected=0x50)       
            check_and_print("ucEndMarkers[1]","u8", expected=0x61)
            check_and_print("ucEndMarkers[2]","u8", expected=0x44)
            check_and_print("ucEndMarkers[3]","u8", expected=0x50)
            check_and_print("ulChecksum","u32")

        if (entryType == DFM_ENTRY_TYPE_PAYLOAD_CHUNK):
            if (verbose_mode): print("  PAYLOAD_CHUNK data:")
            check_and_print(None,"data", length=data_size)
            
        print_indent = 2

        check_and_print("cEndMarker[0]","u8", expected=0xD4)       
        check_and_print("cEndMarker[1]","u8", expected=0xD3)
        check_and_print("cEndMarker[2]","u8", expected=0xD2)
        check_and_print("cEndMarker[3]","u8", expected=0xD1)

        if (verbose_mode):
            print("---------------------------------------------------------------------------")

        alert_data = alert_data[:-4]  # remove the endmarker from the alert data.

        output_path = os.path.join(args.folder, topic_string)
        
        folder = Path(output_path).parent
        folder.mkdir(parents=True, exist_ok=True)

        with open(output_path, 'wb') as f:
            f.write(alert_data)
  
        counter += 1



                       
