@echo off
setlocal ENABLEEXTENSIONS ENABLEDELAYEDEXPANSION

REM Percepio Detect Receiver for Windows, version 2025.2
REM
REM Reads a log file (text or binary) from the DFM library, i.e. the target-side 
REM component of Percepio Detect. It identifies any alert data from DFM
REM and stores it as alert files, intended for ingestion by the Percepio Detect server.
REM
REM Python is required to use this tool.
REM
REM Usage:
REM    percepio-receiver.bat <inputtype> [other args, see below...]
REM
REM    Arguments:
REM     <inputtype> := txt | bin  The type of data: txt = output format of "Serial" module, bin = binary DFM data (e.g. from ITM module).
REM     --inputfile <file>        The file to read, containing DFM data in the specified format (bin or txt).
REM     --folder <dir>            Where the alert files should be saved (the server's alert directory)
REM     --eof <wait|exit>         What happens when reaching end of file: exit (default) or wait for more data (then exit by Ctrl-C).
REM     --device_name <string>    If device name is not specified in DFM, set it here.
REM     --verbose                 Shows (very) verbose output, useful for troubleshooting.
REM

set "SCRIPT_DIR=%~dp0"

if "%~1"=="" goto :help
set "INPUTTYPE=%~1"

REM Parse remaining args into ARG_* variables (no quotes)
call :parse_known_args %*

if /I "%INPUTTYPE%"=="bin" (
  python "%SCRIPT_DIR%bin2alerts.py" %ARG_INPUTFILE% %ARG_FOLDER% %ARG_EOF% %ARG_DEVICE_NAME% %ARG_VERBOSE% 
  exit /b %ERRORLEVEL%
) else if /I "%INPUTTYPE%"=="txt" (
  python "%SCRIPT_DIR%txt2bin.py" %ARG_INPUTFILE% %ARG_EOF% | python "%SCRIPT_DIR%bin2alerts.py" %ARG_FOLDER% %ARG_EOF% %ARG_DEVICE_NAME% %ARG_VERBOSE% 
  exit /b %ERRORLEVEL%
) else (
  echo Error: invalid inputtype "%INPUTTYPE%" (expected bin or txt)
  goto :help
)

:help
echo Percepio Detect Receiver for Windows, version 2025.2
echo Extracts DFM alert files from a device log file or binary file, for ingestion into Percepio Detect.
echo Usage:
echo   percepio-receiver.bat ^<inputtype^> [args...]
echo.
echo   Arguments:
echo     ^<inputtype^> := txt ^| bin  The type of data: txt = output format of "Serial" module, bin = binary DFM data (e.g. from ITM module).
echo     --inputfile ^<file^>        The file to read, containing DFM data in the specified format (bin or txt).
echo     --folder ^<dir^>            Where the alert files should be saved (the server's alert directory)
echo     --eof ^<wait^|exit^>         What happens when reaching end of file: exit (default) or wait for more data (then exit by Ctrl-C).
echo     --device_name ^<string^>    If device name is not specified in DFM, set it here.
echo     --verbose                 Shows (very) verbose output, useful for troubleshooting.
exit /b 0

REM ======================== subroutine =========================
:parse_known_args
REM Drops the first token (<inputtype>) and fills ARG_* vars.
REM Each var is either empty or exactly:   --key value   (no quotes)
shift

REM reset outputs
set "ARG_INPUTFILE="
set "ARG_OUTPUTFILE="
set "ARG_FOLDER="
set "ARG_EOF="
set "ARG_DEVICE_NAME="
set "ARG_VERBOSE="

:next
if "%~1"=="" goto :eof
set "K=" & set "V="

REM split --key=value if present
for /f "tokens=1* delims==" %%A in ("%~1") do ( set "K=%%~A" & set "V=%%~B" )

REM normalize short forms
if /I "!K!"=="-i" set "K=--inputfile"
if /I "!K!"=="-o" set "K=--outputfile"
if /I "!K!"=="-f" set "K=--folder"
if /I "!K!"=="-e" set "K=--eof"
if /I "!K!"=="-d" set "K=--device_name"
if /I "!K!"=="-v" set "K=--verbose"

REM helper: peek at next token (for --key value forms)
set "NEXT=%~2"

REM ---- options with a value ----
if /I "!K!"=="--inputfile" (
  if defined V (set "ARG_INPUTFILE=--inputfile !V!" & shift & goto next)
  if defined NEXT if not "!NEXT:~0,1!"=="-" (set "ARG_INPUTFILE=--inputfile %~2" & shift & shift & goto next)
  shift & goto next
)

if /I "!K!"=="--outputfile" (
  if defined V (set "ARG_OUTPUTFILE=--outputfile !V!" & shift & goto next)
  if defined NEXT if not "!NEXT:~0,1!"=="-" (set "ARG_OUTPUTFILE=--outputfile %~2" & shift & shift & goto next)
  shift & goto next
)

if /I "!K!"=="--folder" (
  if defined V (set "ARG_FOLDER=--folder !V!" & shift & goto next)
  if defined NEXT if not "!NEXT:~0,1!"=="-" (set "ARG_FOLDER=--folder %~2" & shift & shift & goto next)
  shift & goto next
)

if /I "!K!"=="--eof" (
  if defined V (set "ARG_EOF=--eof !V!" & shift & goto next)
  if defined NEXT if not "!NEXT:~0,1!"=="-" (set "ARG_EOF=--eof %~2" & shift & shift & goto next)
  shift & goto next
)

if /I "!K!"=="--device_name" (
  if defined V (set "ARG_DEVICE_NAME=--device_name !V!" & shift & goto next)
  if defined NEXT if not "!NEXT:~0,1!"=="-" (set "ARG_DEVICE_NAME=--device_name %~2" & shift & shift & goto next)
  shift & goto next
)

REM ---- flag (no value) ----
if /I "!K!"=="--verbose" (
  set "ARG_VERBOSE=--verbose"
  shift
  goto next
)

REM unknown token: skip it
shift
goto next
