#!/usr/bin/env bash

# Percepio Detect Receiver for Linux, version 2025.2
#
# Reads a log file (text or binary) from the DFM library, i.e. the target-side 
# component of Percepio Detect. It identifies any alert data from DFM
# and stores it as alert files, intended for ingestion by the Percepio Detect server.
#
# Python is required to use this tool.
#
# Usage:
#    percepio-receiver.sh <inputtype> [other args, see below...]
#
#    Arguments:
#     <inputtype> := txt | bin  The type of data: txt = output format of "Serial" module, bin = binary DFM data (e.g. from ITM module).
#     --inputfile <file>        The file to read, containing DFM data in the specified format (bin or txt).
#     --folder <dir>            Where the alert files should be saved (the server's alert directory)
#     --eof <wait|exit>         What happens when reaching end of file: exit (default) or wait for more data (then exit by Ctrl-C).
#     --device_name <string>    If device name is not specified in DFM, set it here.
#     --verbose                 Shows (very) verbose output, useful for troubleshooting.

echo "Percepio Detect Receiver for Linux, version 2025.2"
echo ""

usage() {
    echo "Percepio Detect Receiver for Linux, version 2025.2"
    echo "Extracts DFM alert files from a device log file or binary file, for ingestion into Percepio Detect."
    echo "Usage:"
    echo "  percepio-receiver.bat <inputtype> [args...]"
    echo ""
    echo "  Arguments:"
    echo "   <inputtype> := txt | bin   The type of data: txt = output format of "Serial" module, bin = binary DFM data (e.g. from ITM module)."
    echo "   -i, --inputfile <file>     The file to read, containing DFM data in the specified format (bin or txt)."
    echo "   -f, --folder <dir>         Where the alert files should be saved (the server's alert directory)."
    echo "   -e, --eof <wait|exit>      What happens when reaching end of file: exit (default) or wait for more data (then exit by Ctrl-C)."
    echo "   -d, --device_name <string> If device name is not specified in DFM, set it here."
    echo "   -v, --verbose              Shows (very) verbose output, useful for troubleshooting."
    exit 1
}

INPUTTYPE=""
ARG_INPUTFILE=""
ARG_FOLDER=""
ARG_EOF=""
ARG_DEVICE_NAME=""
ARG_VERBOSE=""

while [[ $# -gt 0 ]]; do
    case "$1" in

        txt)
            INPUTTYPE="txt"
            ;;

        bin)
            INPUTTYPE="bin"
            ;;

        -i|--inputfile)
            # takes a value
            if [[ -n "$2" && "$2" != --* ]]; then
                ARG_INPUTFILE="--inputfile $2"
                shift
            else
                echo "Error: Missing value for argument $1." >&2
                exit 1
            fi
            ;;

        -f|--folder)
            # takes a value
            if [[ -n "$2" && "$2" != --* ]]; then
                ARG_FOLDER="--folder $2"
                shift
            else
                echo "Error: Missing value for argument $1." >&2
                exit 1
            fi
            ;;

         -e|--eof)
            # takes a value
            if [[ -n "$2" && "$2" != --* ]]; then
                ARG_EOF="--eof $2"
                shift
            else
                echo "Error: Missing value for argument $1." >&2
                exit 1
            fi
            ;;

          -d|--device_name)
            # takes a value
            if [[ -n "$2" && "$2" != --* ]]; then
                ARG_DEVICE_NAME="--device_name $2"
                shift
            else
                echo "Error: Missing value for argument $1." >&2
                exit 1
            fi
            ;;

        -v|--verbose)
            ARG_VERBOSE="--verbose"            
            ;;

        --) # end of arguments
            shift
            break
            ;;

        *)
            echo "Unknown option: $1" >&2
            exit 1
            ;;
    esac
    shift
done

if [[ -n $ARG_VERBOSE ]]; then
    echo "Data type: $INPUTTYPE"
    echo "Reading from: $ARG_INPUTFILE"
    echo "Output to: $ARG_FOLDER"
    echo "Wait or exit at eof: $ARG_EOF"
    echo "Set device_name: $ARG_DEVICE_NAME"
fi

if [[ -z $INPUTTYPE ]]; then
    echo "Missing inputtype (bin|txt)"
    echo ""
    usage
fi

if [[ -z $ARG_INPUTFILE ]]; then
    echo "--inputfile not set"
    echo ""
    usage
fi

if [[ -z $ARG_FOLDER ]]; then
    echo "--folder not set"
    echo ""
    usage
fi


SCRIPT_DIR="$(cd -- "$(dirname -- "${BASH_SOURCE[0]}")" >/dev/null 2>&1 && pwd)"

if [[ "$EOF" == "wait" ]]; then
echo " Quit using Ctrl-C"
fi

if [[ "$INPUTTYPE" == "bin" ]]; then

  python3 "$SCRIPT_DIR/bin2alerts.py" $ARG_INPUTFILE $ARG_FOLDER $ARG_EOF $ARG_DEVICE_NAME $ARG_VERBOSE

elif [[ "$INPUTTYPE" == "txt" ]]; then

  python3 "$SCRIPT_DIR/txt2bin.py" $ARG_INPUTFILE $ARG_EOF | python3 "$SCRIPT_DIR/bin2alerts.py" $ARG_FOLDER $ARG_EOF -d "$DEVICE_ID" $ARG_DEVICE_NAME $ARG_VERBOSE

else

  echo "INPUTTYPE not provided or invalid value. Expected bin or txt." >&2
  exit 2

fi



