#!/usr/bin/python3

import sys
import argparse
import re
from binascii import unhexlify
import enum
import time
from typing import Tuple

class ChunkResultType(enum.Enum):
    Start = enum.auto()
    Data = enum.auto()
    End = enum.auto()
    NotDfm = enum.auto()

class DataBlockParseState(enum.Enum):
    NotRunning = enum.auto()
    Parsing = enum.auto()

def crc16_ccit(data: bytearray):
    seed = 0x0000
    for i in range(0, len(data)):
        cur_byte = data[i]
        e = (seed ^ cur_byte) & 0xFF
        f = (e ^ (e << 4)) & 0xFF
        seed = (seed >> 8) ^ (f << 8) ^ (f << 3) ^ (f >> 4)

    return seed

def info_log(message):
    sys.stderr.write("{}\n".format(message))

class DatablockParser:

    @staticmethod
    def process_line(line: str) -> Tuple[ChunkResultType, None or bytes or int]:
        """
        Process a line and return what it was so that the state machine can handle it properly
        :param line:
        :return:
        """

        line = line.strip()
        # Normal log line, ignore
        if not (line.startswith("[[ ") and line.endswith(" ]]")):
            return ChunkResultType.NotDfm, None

        line = re.sub(r"^\[\[\s", "", line)
        line = re.sub(r"\s\]\]$", "", line)

        if line == "DevAlert Data Begins":
            return ChunkResultType.Start, None
        elif line.startswith("DATA: "):
            line = re.sub(r"^DATA:\s", "", line)
            return ChunkResultType.Data, unhexlify(line.replace(" ", ""))
        elif matches := re.match(r'^DevAlert\sData\sEnded.\sChecksum:\s(\d{1,5})', line):
            return ChunkResultType.End, int(matches.group(1))
        else:
            return ChunkResultType.NotDfm, None

class LineParser:

    def __init__(self):
        self.line_buffer = ""
        self.parsing = False

    def process(self, line_data) -> bool:
        stripped_line = line_data.replace("\n", "")
        if not self.parsing:
            self.line_buffer = ""
            if stripped_line.startswith("[[") and stripped_line.endswith("]]"):
                self.line_buffer = stripped_line
                return True
            elif stripped_line.startswith("[["):
                self.line_buffer += stripped_line
                self.parsing = True
                return False
            elif stripped_line.startswith("[") and len(stripped_line) < 2:
                self.line_buffer += stripped_line
                self.parsing = True
                return False
            else:
                return False
        elif self.parsing:
            # Apparently we've encountered a new line
            if stripped_line.startswith("[[") and stripped_line.endswith("]]"):
                self.line_buffer = stripped_line
                self.parsing = False
                return True
            elif stripped_line.startswith("[["):
                self.line_buffer = stripped_line
                return False
            elif stripped_line.endswith("]]"):
                self.line_buffer += stripped_line
                self.parsing = False
                return True
            else:
                self.line_buffer += stripped_line
                # Invalid line which started with [, drop it
                if not self.line_buffer.startswith("[["):
                    self.line_buffer = ""
                    self.parsing = False
                return False

from argparse import RawTextHelpFormatter

if __name__ == "__main__":
    parser = argparse.ArgumentParser(
        prog='txt2bin',
        description='Extracts DFM data from device log files (text files) and writes it to binary file (by default named dfmdata.bin), intended for further processing by bin2alerts.py to generate alert files .\nThe input file is assumed to originate from the DFM \"Serial\" module, that produces data like this:\n  [[ DevAlert Data Begins ]]\n  [[ DATA: D1 D2 D3 D4 F0 0F ... ]]\n  [[ DATA: ... ]]\n  [[ DevAlert Data Ended. Checksum: 0 ]]\nIncomplete DFM entries are discarded.\n',
        formatter_class=RawTextHelpFormatter     
    )
    parser.add_argument('-i', '--inputfile', type=str, help='The log file to read, containing DFM data.')
    parser.add_argument('-o', '--outputfile', type=str, help='Output file. Defaults to stdout if not provided.')
    parser.add_argument('-e', '--eof', type=str, help='What to do at End Of File (EOF):\n    wait: keeps waiting for more data (exit using Ctrl-C).\n    exit: exits directly at end of file (default).', required=False, default='exit')
    
    # Tolerated but ignored args, allows for passing bin2alerts and txt2bin the same args from a shared top-level script.
    parser.add_argument('-v', '--verbose',     action='store_true', help='Not used. Added for argument compatibility between scripts.', required=False)    
    parser.add_argument('-d', '--device_name', type=str, help='Not used. Added for argument compatibility between scripts.', required=False)
    parser.add_argument('-f', '--folder',      type=str, help='Not used. Added for argument compatibility between scripts.', required=False)
    
    args = parser.parse_args()

    outputfile = args.outputfile

    line_parser = LineParser()
    block_parse_state = DataBlockParseState.NotRunning
    accumulated_payload = bytes([])
    
    # Create and clear the output file, only if writing to a real file
    if outputfile is not None:
        open(outputfile, "wb").close()

    with open(args.inputfile, "r", encoding="latin-1", buffering=1) as fh:
        while 1:
            line = fh.readline()

            # We've reached the end of the file, just sleep and try reading again.
            # This makes it possible to continuously read the file
            if line == "":
                if args.eof!="wait":
                    # We've reached the end of the file, and we just wanted to do that
                    break
                time.sleep(1)
                continue

            # If parsing in the middle of data being written to file, try to get the whole line
            if not line_parser.process(line):
                continue

            parse_result, payload = DatablockParser.process_line(line_parser.line_buffer)

            if block_parse_state == DataBlockParseState.NotRunning:
                if parse_result == ChunkResultType.Start:
                    block_parse_state = DataBlockParseState.Parsing
                    accumulated_payload = bytes([])
                elif parse_result == ChunkResultType.NotDfm:
                    info_log("Got unexpected or corrupted data, not in the expected format: {}".format(line_parser.line_buffer))
                    continue
                else:
                    info_log("Got {} without a Data Start label, incomplete log?".format(parse_result.name))
            elif block_parse_state == DataBlockParseState.Parsing:

                if parse_result == ChunkResultType.Data:
                    accumulated_payload += payload                    
              
                elif parse_result == ChunkResultType.Start:
                    info_log("Got a start while parsing, resetting payload")
                    accumulated_payload = bytes([])
  
                elif parse_result == ChunkResultType.End:
                    block_parse_state = DataBlockParseState.NotRunning

                    if len(accumulated_payload) == 0:
                        info_log("Got empty message")
                    else:
                        # No checksum provided, skip check
                        if payload != 0:
                           calculated_crc = crc16_ccit(bytearray(accumulated_payload))
                           if calculated_crc != payload:
                               info_log("Got crc mismatch, calculated: {}, got: {}, payload length: {}".format(
                                   calculated_crc, payload, len(accumulated_payload)))
                               continue

                        # A full DFM entry has been received, append it.
                        
                        # with open(outputfile, 'ab') as file:
                          #  file.write(accumulated_payload)

                        if outputfile is None:
                            # write to stdout (binary)
                            try:
                                sys.stdout.buffer.write(accumulated_payload)
                                sys.stdout.buffer.flush()
                            except BrokenPipeError:
                                # downstream closed (e.g., consumer exited) -> exit quietly
                                sys.exit(0)
                        else:
                            with open(outputfile, "ab") as f:
                                f.write(accumulated_payload)

