#!/bin/bash

# The parameter ALERTS_DIR should specify the folder containing the alert files.
# Note that Percepio Client uses a separate ALERTS_DIR setting that should point
# to the same directory. The included demo data directory is used by default. 
# To use a different ALERTS_DIR setting, uncomment the line below.

# ALERTS_DIR="..."

# Default ALERTS_DIR, containing demo data.
DEFAULT_ALERTS_DIR="$(pwd)/../test-data/alert-files"

# Specify your Detect Server license key below, e.g. $LICENSE="ABCD-ABCD-ABCD-ABCD"
LICENSE=""

# When used on a shared server, change this to the server IP address (as seen from the clients).
BACKEND_IP="127.0.0.1"

# Below variables can be changed if needed, e.g. if some port is busy.
HTTP_FRONTEND_PORT=8080
HTTPS_FRONTEND_PORT=8081
HTTP_BACKEND_PORT=7074
HTTPS_BACKEND_PORT=7073

if [ -z "${ALERTS_DIR}" ]; then
ALERTS_DIR=$DEFAULT_ALERTS_DIR
echo "ALERTS_DIR: Using demo data, $(realpath $ALERTS_DIR)."
fi

# Translates to an absolute path.
ALERTS_DIR=$(realpath $ALERTS_DIR)

check_path() {
  dir_to_check=$1
  label=$2
  if [ -z "$dir_to_check" ]; then
    echo "Checking $label: Error, parameter $label not set."
    exit 1
  fi  
  if [ ! -d "$dir_to_check" ]; then
    echo "Checking $label: Error, not a valid path."
    echo "$label: $dir_to_check"
    exit 1
  fi 
  echo "Checking $label: OK"
}

# Checks that ALERTS_DIR path is valid.
check_path $ALERTS_DIR "ALERTS_DIR"

check_path_access_rights() {
  local dir_to_check=$1

  # Check if the directory is writable for others
  if [ -d "$dir_to_check" ] && [ -w "$dir_to_check" ] && [ "$(stat -c %A "$dir_to_check" | cut -c 9)" = "w" ]; then
      echo "The directory '$dir_to_check' is writable for others."
  else
      # Print error message in red and exit
      echo -e "\033[1;31mThe directory '$dir_to_check' is NOT writable for others. Please change the permissions using:\033[0m"
      echo -e "\033[1;31mchmod o+rw '$dir_to_check'\033[0m"
      exit 1
  fi
}

# check if ALERTS_DIR is writable for others
check_path_access_rights "$ALERTS_DIR"


BACKEND_REST_API="http://$BACKEND_IP:$HTTP_BACKEND_PORT/api"

# Container names
DATABASE_CONTAINER="detect-database"
ALERT_SENDER_CONTAINER="detect-alert-sender"
FRONTEND_CONTAINER="detect-frontend"
BACKEND_CONTAINER="detect-backend"

# Image tag
TAG="2025.2"

# Image names
FRONTEND_IMAGE="percepio/detect-frontend:$TAG"
BACKEND_IMAGE="percepio/detect-backend:$TAG"
ALERT_SENDER_IMAGE="percepio/detect-alertsender:$TAG"
DATABASE_IMAGE="percepio/detect-database:$TAG"

# MySQL variables
MYSQL_ROOT_PASSWORD="DevalertMysql"
MYSQL_DATABASE="devalert"

# Docker network
NETWORK_NAME="percepio-network"

# Volume name
VOLUME_NAME="percepio_database"

BACKEND_ALERTS_REST_ENDPOINT="http://$BACKEND_CONTAINER:$HTTP_BACKEND_PORT/api/alert"

# pull images from dockerhub
pull_images() {
  echo "Pull Docker containers..."
  docker pull $FRONTEND_IMAGE
  docker pull $BACKEND_IMAGE
  docker pull $ALERT_SENDER_IMAGE
  docker pull $DATABASE_IMAGE
}

# Function to start Docker containers
start_containers() {
  # pull_images
  echo "Starting Docker containers..."

  # Create a custom network if it doesn't exist
  docker network create $NETWORK_NAME || true

  # Ensure the Docker volume exists
  docker volume create $VOLUME_NAME || true

  # Start Database container
   docker run -d \
    --name $DATABASE_CONTAINER \
    --network $NETWORK_NAME \
    -e MYSQL_ROOT_PASSWORD=$MYSQL_ROOT_PASSWORD \
    -e MYSQL_DATABASE=$MYSQL_DATABASE \
    -v $VOLUME_NAME:/var/lib/mysql \
    $DATABASE_IMAGE

  # Start the Backend container
  docker run -d \
    --name $BACKEND_CONTAINER \
    --network $NETWORK_NAME \
    --link $DATABASE_CONTAINER:mysql \
    -p $HTTP_BACKEND_PORT:$HTTP_BACKEND_PORT \
    -p $HTTPS_BACKEND_PORT:$HTTPS_BACKEND_PORT \
    -e LICENSE_KEY=$LICENSE \
    -e RDS_ENDPOINT=mysql \
    -e RDS_PORT=3306 \
    -e RDS_DATABASE=$MYSQL_DATABASE \
    -e RDS_USER=root \
    -e RDS_PASSWORD=$MYSQL_ROOT_PASSWORD \
    $BACKEND_IMAGE

  # Start the Frontend container
  docker run -d \
    --name $FRONTEND_CONTAINER \
    --network $NETWORK_NAME \
    -p $HTTP_FRONTEND_PORT:80 \
    -p $HTTPS_FRONTEND_PORT:443 \
    -e VUE_APP_API_ENDPOINT=$BACKEND_REST_API \
    -v ./cert/detect.crt:/etc/nginx/ssl/detect.crt \
    -v ./cert/detect.key:/etc/nginx/ssl/detect.key \
    $FRONTEND_IMAGE

  # Start the Alert Sender container
  docker run -d \
    --name $ALERT_SENDER_CONTAINER \
    --network $NETWORK_NAME \
    -e SEND_ALERTS_URL=$BACKEND_ALERTS_REST_ENDPOINT \
    -e TRACE_DIR=/app/data/ \
    -e CHECK_INTERVAL_MS=5000 \
    -v $ALERTS_DIR:/app/data \
    --tty \
    --restart on-failure \
    $ALERT_SENDER_IMAGE

  echo "Percepio Detect server started."

}

# Function to stop Docker containers and clean up
stop_containers() {
  echo "Stopping and removing Docker containers..."

  docker stop $ALERT_SENDER_CONTAINER $FRONTEND_CONTAINER $BACKEND_CONTAINER $DATABASE_CONTAINER
  docker rm $ALERT_SENDER_CONTAINER $FRONTEND_CONTAINER $BACKEND_CONTAINER $DATABASE_CONTAINER

  echo "Removing custom network..."
  docker network rm $NETWORK_NAME

  echo "All containers stopped and removed, network '$NETWORK_NAME' deleted."
}

# Function to follow logs for all containers
follow_logs() {
  docker logs -f $DATABASE_CONTAINER &
  docker logs -f $BACKEND_CONTAINER &
  docker logs -f $FRONTEND_CONTAINER &
  docker logs -f $ALERT_SENDER_CONTAINER &
  wait
}

# Function to stop and clean up Docker resources with a confirmation prompt
cleanup_docker_resources() {
  # Display warning and prompt for user confirmation
  echo -e "\033[1;33mWARNING: This cleanup will delete existing alert data from the database and remove Docker containers, network, and volume.\033[0m"
  read -p "Do you want to proceed? Type 'yes' to confirm: " confirmation

  if [ "$confirmation" != "yes" ]; then
    echo "Cleanup aborted by user."
    return
  fi

  echo "Starting full Docker cleanup..."

  # Reuse stop_containers function to stop and remove all containers
  stop_containers

  # Remove Docker network if it exists
  if docker network ls --filter name=^$NETWORK_NAME$ | grep -q $NETWORK_NAME; then
    echo "Removing Docker network..."
    docker network rm $NETWORK_NAME
  fi

  # Remove Docker volume if it exists
  if docker volume ls --filter name=^$VOLUME_NAME$ | grep -q $VOLUME_NAME; then
    echo "Removing Docker volume..."
    docker volume rm $VOLUME_NAME
  fi

  rm -f ${ALERTS_DIR}/failure.txt
  rm -f ${ALERTS_DIR}/success.txt
  rm -f ${ALERTS_DIR}/lasttrace.txt

  echo "Cleanup complete: Containers stopped, network and volume removed."
}

# Main script logic
case "$1" in
  start)
    start_containers
    ;;
  stop)
    stop_containers
    ;;
  logs)
    follow_logs
    ;;
  cleanup)
    cleanup_docker_resources
    ;;
  *)
    echo "Usage: $0 {start|stop|logs|cleanup}"
    exit 1
    ;;
esac
